(*
  This file is a part of New Audio Components package 1.1
  Copyright (c) 2002-2007, Andrei Borovsky. All rights reserved.
  See the LICENSE file for more details.
  You can contact me at anb@symmetrica.net
*)

(* $Revision: 1.9 $ $Date: 2007/07/10 07:36:23 $ *)

unit ACS_Classes;

interface

uses

{$IFDEF WIN32}
  Windows,
{$ENDIF}
  Classes, SysUtils, SyncObjs;

type

  TOutputStatus = (tosPlaying, tosPaused, tosIdle);

  TFileOutputMode = (foRewrite = 0, foAppend);

  TOutputFunc = function(Abort : Boolean):Boolean of object;

  //TThreadDoneEvent = procedure of object;

  TThreadExceptionEvent = procedure(Sender : TComponent) of object;

  THandleException = procedure(Sender : TComponent; const Msg : String) of object;

  TOutputDoneEvent = procedure(Sender : TComponent) of object;

  TOutputProgressEvent = procedure(Sender : TComponent) of object;

{$IFDEF LINUX}
// File access mask constants
const

  famUserRead = 64;
  famUserWrite = 128;
  famGroupRead = 8;
  famGroupWrite = 16;
  famOthersRead = 1;
  famOthersWrite = 2;

{$ENDIF}

type

  EAuException = class(Exception)
  end;

  TAuThread = class(TThread)
  private
    ErrorMsg : String;
    procedure CallOnProgress;
    procedure CallOnException;
//    procedure CallOnDone;
    procedure RaiseDoneEvent;
  public
    DoNotify : Boolean; // Flag that determines if OnDone should be raised a the end of output
                        // default value is True, may be set to Dalse in Stop method.
    Stopped : Boolean;  // Flag that tells when the output is actually stopped.
                        // Used when DoNotify is set to False.
    Parent : TComponent;
    bSuspend : Boolean;
    //Terminating : Boolean;
    Stop : Boolean;
    HandleException : THandleException;
    Delay : Integer;
{$IFDEF WIN32}
    CS : TRTLCriticalSection;
{$ENDIF}
    procedure Execute; override;
  end;

  TVerySmallThread = class(TThread)
  public
    FOnDone  : TOutputDoneEvent;
    Sender : TComponent;
    procedure Execute; override;
    procedure CallOnDone;
  end;

  TAuInput = class(TComponent)
  protected
    FPosition : Integer;
    FSize : Integer;
    Busy : Boolean;
    BufStart, BufEnd : Integer;
    (* We don't declare the buffer variable here
     because different descendants may need different buffer sizes *)
    function GetBPS : Integer; virtual; abstract;
    function GetCh : Integer; virtual; abstract;
    function GetSR : Integer; virtual; abstract;
    function GetTotalTime : Integer; virtual;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure GetData(var Buffer : Pointer; var Bytes : Integer); virtual; abstract;
    function CopyData(Buffer : Pointer; BufferSize : Integer) : Integer;
    procedure Reset; virtual;
    procedure Init; virtual; abstract;
    procedure Flush; virtual; abstract;
    property BitsPerSample : Integer read GetBPS;
    property Position : Integer read FPosition;
    property SampleRate : Integer read GetSR;
    property Channels : Integer read GetCh;
    property Size : Integer read FSize;
    property TotalTime : Integer read GetTotalTime;
  end;

  TAuOutput = class(TComponent)
  protected
    FExceptionMessage : String;
    CanOutput : Boolean;
    CurProgr : Integer;
    Thread : TAuThread;
    FInput : TAuInput;
    FOnDone : TOutputDoneEvent;
    FOnProgress : TOutputProgressEvent;
    Busy : Boolean;  // Set to true by Run and to False by WhenDone.
    FOnThreadException : TThreadExceptionEvent;
   // InputLock : Boolean;
    function GetPriority : {$IFDEF LINUX} Integer; {$ENDIF} {$IFDEF WIN32} TThreadPriority; {$ENDIF}
    function GetSuspend : Boolean;
    function GetProgress : Integer;
    procedure SetInput(vInput : TAuInput); virtual;
    procedure SetPriority(Priority : {$IFDEF LINUX} Integer {$ENDIF} {$IFDEF WIN32} TThreadPriority {$ENDIF});
    procedure SetSuspend(v : Boolean);
    procedure WhenDone; // Calls descendant's Done method
    function GetTE : Integer;
    function GetStatus : TOutputStatus;
    function DoOutput(Abort : Boolean):Boolean; virtual; abstract;
    procedure Done; virtual; abstract; // Calls FInput.Flush
    procedure Prepare; virtual; abstract; // Calls FInput.init
    function GetDelay : Integer;
    procedure SetDelay(Value : Integer);
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure HandleException(Sender : TComponent; const ErrorMessage : String);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    {$IFDEF WIN32}
    procedure Abort;
    {$ENDIF}
    procedure Pause;
    procedure Resume;
    procedure Run;
    procedure Stop(Notify : Boolean = True);
    procedure WaitForStop;
    property Delay : Integer read GetDelay write SetDelay;
    property ThreadPriority : {$IFDEF LINUX} Integer {$ENDIF} {$IFDEF WIN32} TThreadPriority {$ENDIF} read GetPriority write SetPriority;
    property Progress : Integer read GetProgress;
    property Status : TOutputStatus read GetStatus;
    property TimeElapsed : Integer read GetTE;
    property ExceptionMessage : String read FExceptionMessage;
  published
    property Input : TAuInput read Finput write SetInput;
    property SuspendWhenIdle : Boolean read GetSuspend write SetSuspend;
    property OnDone : TOutputDoneEvent read FOnDone write FOndone;
    property OnProgress : TOutputProgressEvent read FOnProgress write FOnProgress;
    property OnThreadException : TThreadExceptionEvent read FOnThreadException write FOnThreadException;
  end;

  TAuStreamedInput = class(TAuInput)
  protected
    FStream : TStream;
    FStreamAssigned : Boolean;
    FSeekable : Boolean;
    procedure SetStream(aStream : TStream);
  public
    property Seekable : Boolean read FSeekable write FSeekable;
    property Stream : TStream read FStream write SetStream;
    constructor Create(AOwner: TComponent); override;
  end;

  TAuStreamedOutput = class(TAuOutput)
  protected
    FStream : TStream;
    FStreamAssigned : Boolean;
    procedure SetStream(aStream : TStream);
  public
    property Stream : TStream read FStream write SetStream;
  end;

  TAuFileIn = class(TAuStreamedInput)
  protected
    DataCS : TCriticalSection;
    FFileName : TFileName;
    FOpened : Integer;
    FValid : Boolean;
    FBPS, FSR, FChan : Integer;
    FTime : Integer;
    FLoop : Boolean;
    FStartSample, FEndSample : Integer;
    FTotalSamples : Integer;
    function GetBPS : Integer; override;
    function GetCh : Integer; override;
    function GetSR : Integer; override;
    function GetTime : Integer;
    function GetValid : Boolean;

    (* Note on FSize calculation:
      FSize is calculated in OpenFile method as the FULL file size.
      More precise calculations regarding StartSample/EndSample are done in Init. *)

    procedure OpenFile; virtual; abstract;
    procedure CloseFile; virtual; abstract;
    function GetTotalTime : Integer; override;
    procedure Reset; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Flush; override;
    procedure Init; override;
    function Seek(SampleNum : Integer) : Boolean; virtual; abstract;
    function SetStartTime(Minutes, Seconds : Integer) : Boolean;
    function SetEndTime(Minutes, Seconds : Integer) : Boolean;
    procedure Jump(Offs : Integer);
    property Time : Integer read GetTime;
    property TotalSamples : Integer read FTotalSamples;
    property Valid : Boolean read GetValid;
  published
    property EndSample : Integer read FEndSample write FEndSample;
    property FileName : TFileName read FFileName write FFileName stored True;
    property Loop : Boolean read FLoop write FLoop;
    property StartSample : Integer read FStartSample write FStartSample;
  end;

  TAuFileOut = class(TAuStreamedOutput)
  protected
    FFileName : TFileName;
    FFileMode : TFileOutputMode;
    FAccessMask : Integer;
    procedure SetFileMode(aMode : TFileOutputMode); virtual;
  public
    constructor Create(AOwner: TComponent); override;
{$IFDEF LINUX}
    property AccessMask : Integer read FAccessMask write FAccessMask;
{$ENDIF}
  published
    property FileMode : TFileOutputMode read FFileMode write SetFileMode;
    property FileName : TFileName read FFileName write FFileName;
  end;

  TAuConverter = class(TAuInput)
  protected
    FInput : TAuInput;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure SetInput(aInput : TAuInput); virtual;
  published
    property Input : TAuInput read FInput write SetInput;
  end;

const

  STREAM_BUFFER_SIZE = $80000;

type

  // Circular buffer with TStream interface

  TACSBufferMode = (bmBlock, bmReport);

implementation

  constructor TAuInput.Create;
  begin
    inherited Create(AOwner);
  end;

  destructor TAuInput.Destroy;
  begin
    inherited Destroy;
  end;

  procedure TAuThread.RaiseDoneEvent;
  var
    DoneThread : TVerySmallThread;
  begin
     // This ensures that OnDone event is called outside this thread
     DoneThread := TVerySmallThread.Create(True);
     DoneThread.Sender := TAuOutput(Parent);
     DoneThread.FOnDone := TAuOutput(Parent).FOnDone;
     DoneThread.FreeOnTerminate := True;
     DoneThread.Resume;
   end;

  procedure TAuThread.Execute;
  var
    ParentComponent : TAuOutput;
    Res : Boolean;
  begin
    ParentComponent := TAuOutput(Parent);
    while not Terminated do
    begin
      {$IFDEF WIN32}
      EnterCriticalSection(CS);
      {$ENDIF}
      if Delay > 5 then sleep(Delay);
      try
        if not Stop then
          if ParentComponent.Progress <> ParentComponent.CurProgr then
          begin
            ParentComponent.CurProgr := ParentComponent.Progress;
            if Assigned(ParentComponent.FOnProgress) then Synchronize(CallOnProgress);
          end;
        Res := ParentComponent.DoOutput(Stop);
        if Stop or (not Res) then
        begin
          Stop := False;
          ParentComponent.WhenDone;
          if DoNotify then
            RaiseDoneEvent;
          Stopped := True;
          if bSuspend and (not Terminated) then Self.Suspend;
        end;
      except
        on E : Exception do
        begin
          ParentComponent.WhenDone;
          if DoNotify then
            RaiseDoneEvent;
          Stopped := True; // Stop := False;
          ErrorMsg := E.Message;
          Synchronize(CallOnException);
          if bSuspend and (not Terminated) then Self.Suspend;
        end;
      end;
      {$IFDEF WIN32}
      LeaveCriticalSection(CS);
      {$ENDIF}
    end;
  end;

  constructor TAuOutput.Create;
  begin
    inherited Create(AOwner);
    Thread := TAuThread.Create(True);
    Thread.Parent := Self;
    Thread.DoNotify := True;
    Thread.FreeOnTerminate := True;
    Thread.HandleException := HandleException;
    {$IFDEF WIN32}
      SetSuspend(True);
      InitializeCriticalSection(Thread.CS);
    {$ENDIF}
  end;

  destructor TAuOutput.Destroy;
  begin
    Stop(False);
    WaitForStop;
    {$IFDEF WIN32}
    DeleteCriticalSection(Thread.CS);
    {$ENDIF}
    inherited Destroy;
  end;

  procedure TAuOutput.WhenDone;
  begin
    if not Busy then Exit;
    CanOutput := False;
    Done;
    Busy := False;
  end;

  procedure TAuOutput.Run;
  begin
    FExceptionMessage := '';
    if Busy then raise EAuException.Create('Component is Busy');
    if not Assigned(FInput) then raise EAuException.Create('Input is not assigned');
    try
      Busy := True;
      Prepare;
      Thread.Stop := False;
      CanOutput := True;
      if Thread.Suspended then Thread.Resume;
    except
      on E : Exception do
      begin
        try
          if not Thread.Suspended then
          begin
            Stop;
            WaitForStop;
          end else
          begin
            WhenDone;
            Thread.RaiseDoneEvent;
          end;
        except
        end;
        Busy := False;
        HandleException(Self, E.Message);
      end;
    end;
  end;

  procedure TAuOutput.Stop;
  begin
    Thread.DoNotify := Notify;
    Thread.Stopped := False;
    Thread.Stop := True;
  end;

  procedure TAuOutput.WaitForStop;
  begin
    while (not Thread.Suspended) and (not Thread.Stopped) do
    begin
      if Thread.Delay > 5 then sleep(Delay);
      CheckSynchronize; // to release possible deadlocks
    end;
    Thread.DoNotify := True;
  end;

  function TAuOutput.GetStatus;
  begin
    if Busy then
    begin
      if Self.Thread.Suspended then Result := tosPaused
      else Result := tosPlaying;
    end else Result := tosIdle;
  end;

  procedure TAuOutput.SetPriority;
  begin
    Thread.Priority := Priority;
  end;

  function TAuOutput.GetPriority;
  begin
    Result := Thread.Priority;
  end;

  procedure TAuOutput.SetInput;
  begin
    if Busy then
    begin
      Stop(False);
      WaitForStop;
      FInput := vInput;
      Run;
    end else
    FInput := vInput;
  end;

  function  TAuOutput.GetProgress;
  begin
    if not Assigned(Finput) then
    begin
      Result := 0;
      Exit;
    end;
    case Finput.Size of
      0: Result := 0;
      -1: Result := -1;
      else Result := Round((FInput.Position/FInput.Size)*100);
    end;
  end;

  procedure TAuOutput.Pause;
  begin
    If not Thread.Suspended then Thread.Suspend;
  end;

  procedure TAuOutput.Resume;
  begin
    If Thread.Suspended then Thread.Resume;
  end;

  function TAuOutput.GetSuspend;
  begin
    Result := Thread.bSuspend;
  end;

  procedure TAuOutput.SetSuspend;
  begin
    Thread.bSuspend := v;
  end;

  constructor TAuStreamedInput.Create;
  begin
    inherited Create(AOwner);
    FSeekable := True;
  end;

  function TAuFileIn.GetBPS;
  begin
    if FSeekable then
    begin
      OpenFile;
      Result := FBPS;
      CloseFile;
    end else Result := FBPS;
  end;

  function TAuFileIn.GetCh;
  begin
    if FSeekable then
    begin
      OpenFile;
      Result := FChan;
      CloseFile;
    end else Result := FChan;
  end;

  function TAuFileIn.GetSR;
  begin
    if FSeekable then
    begin
      OpenFile;
      Result := FSR;
      CloseFile;
    end else Result := FSR;
  end;

  function TAuFileIn.GetTime;
  begin
    if FSeekable then
    begin
      OpenFile;
      Result := FTime;
      CloseFile;
    end else Result := FTime;
  end;

  function TAuFileIn.GetValid;
  begin
    if (not FStreamAssigned) and (FileName = '') then
    begin
      Result := False;
    end else
    if FSeekable then
    begin
      OpenFile;
      Result := FValid;
      CloseFile;
    end else Result := FValid;
  end;

  procedure TAuFileIn.Init;
  begin
    if Busy then raise EAuException.Create('The component is Busy');
    if not FStreamAssigned then
    if FFileName = '' then raise EAuException.Create('The file name is not assigned');
    DataCS := TCriticalSection.Create;
    OpenFile; // FTotalSamples should be set here
    if StartSample <> 0 then Seek(StartSample);
    if (StartSample <> 0) or (FEndSample <> -1) then
    begin
      if FEndSample = -1 then
        FTotalSamples :=  FTotalSamples - FStartSample + 1
      else
         FTotalSamples := FEndSample - FStartSample + 1;
    end;
    FSize := FTotalSamples * FChan * (FBPS div 8);
    Busy := True;
    BufStart := 1;
    BufEnd := 0;
    FPosition := 0;
  end;

  procedure TAuFileIn.Flush;
  begin
    CloseFile;
     DataCS.Free;
    Busy := False;
  end;

  procedure TAuFileIn.Jump;
  var
    Curpos : Double;
    Cursample : Integer;
  begin
    if (not FSeekable) or (FSize = 0) then Exit;
    Curpos := FPosition/FSize + offs/100;
    if Curpos < 0 then Curpos := 0;
    if Curpos > 1 then Curpos := 1;
    Cursample := Round(Curpos*FTotalSamples);
    Seek(Cursample);
  end;

  function TAuOutput.GetTE;
  begin
     if not Assigned(FInput) then
     Result := 0
     else
     Result := Round(FInput.Position/((FInput.BitsPerSample shr 3) *FInput.Channels*FInput.SampleRate));
  end;

  function TAuOutput.GetDelay;
  begin
    if Assigned(Thread) then Result := Thread.Delay;
  end;

  procedure TAuOutput.SetDelay;
  begin
    if Assigned(Thread) then
    if Value <= 100 then Thread.Delay := Value;
  end;

  function TAuInput.GetTotalTime;
  begin
    Result := 0;  // Default result for the streams.
  end;

  function TAuFileIn.GetTotalTime;
  begin
    OpenFile;
    if (SampleRate = 0) or (Channels = 0) or (BitsPerSample = 0) then Exit;
    Result := Round(Size/(SampleRate*Channels*(BitsPerSample shr 3)));
    CloseFile;
  end;

  procedure TAuStreamedInput.SetStream;
  begin
    FStream := aStream;
    if FStream <> nil then FStreamAssigned := True
    else FStreamAssigned := False;
  end;

  procedure TAuStreamedOutput.SetStream;
  begin
    FStream := aStream;
    if FStream <> nil then FStreamAssigned := True
    else FStreamAssigned := False;
  end;

  procedure TAuOutput.Notification;
  begin
    // Remove the following two lines if they cause troubles in your IDE
    if (AComponent = FInput) and (Operation = opRemove )
    then Input := nil;
    inherited Notification(AComponent, Operation);
  end;

  procedure TAuInput.Reset;
  begin
    try
      Flush;
    except
    end;
    Busy := False;
  end;

  procedure TAuOutput.HandleException;
  begin
   CanOutput := False;
   Busy := False;
   Self.FExceptionMessage := ErrorMessage;
   if Assigned(FOnThreadException) then FOnThreadException(Self);
  end;

  procedure TAuFileIn.Reset;
  begin
    inherited Reset;
    FOpened := 0;
  end;


  constructor TAuFileOut.Create;
  begin
    inherited Create(AOwner);
    {$IFDEF LINUX}
    FAccessMask := $1B6; // rw-rw-rw-
    {$ENDIF}
  end;

  procedure TAuFileOut.SetFileMode;
  begin
    FFileMode := foRewrite;
  end;

  procedure TAuConverter.Notification;
  begin
    // Remove the following two lines if they cause troubles in your IDE
    if (AComponent = FInput) and (Operation = opRemove )
    then Input := nil;
    inherited Notification(AComponent, Operation);
  end;

  procedure TAuConverter.SetInput;
  begin
    if aInput = Self then Exit;
    if Busy then
    begin
      raise EAuException.Create('Converter components cannot change input on the fly.');
  (*    NewInput := aInput;
      NewInput.Init;
      OldInput := FInput;
      while InputLock do;
      InputLock := True;
      FInput := NewInput;
      InputLock := False;
      OldInput.Flush; *)
    end else
    FInput := aInput;
  end;

  function TAuFileIn.SetStartTime;
  var
    Sample : Integer;
  begin
    Result := False;
    if not FSeekable then Exit;
    OpenFile;
    CloseFile;
    Sample := (Minutes*60+Seconds)*FSR;
    if Sample > FTotalSamples then Exit;
    FStartSample := Sample;
    Result := True;
  end;

  function TAuFileIn.SetEndTime;
  var
    Sample : Integer;
  begin
    Result := False;
    if not FSeekable then Exit;
    OpenFile;
    CloseFile;
    Sample := (Minutes*60+Seconds)*FSR;
    if Sample > FTotalSamples then Exit;
    FEndSample := Sample;
    Result := True;
  end;

  constructor TAuFileIn.Create;
  begin
    inherited Create(AOwner);
    FStartSample := 0;
    FEndSample := -1;
  end;

  destructor TAuFileIn.Destroy;
  begin
    inherited Destroy;
  end;

 procedure TVerySmallThread.Execute;
 var
   Output : TAuOutput;
 begin
   Output := Sender as TAuOutput;
   if Output.Thread.bSuspend then
     while not Output.Thread.Suspended do // To fire OnDone only after the output thread is suspended
       Sleep(Output.Delay);
   Synchronize(CallOnDone);
 end;

 procedure  TVerySmallThread.CallOnDone;
 begin
   if Assigned(FOnDone) then FOnDone(Sender);
 end;

 procedure TAuThread.CallOnProgress;
  var
    ParentComponent : TAuOutput;
  begin
    ParentComponent := TAuOutput(Parent);
    ParentComponent.FOnProgress(ParentComponent);
  end;

  procedure TAuThread.CallOnException;
  begin
    HandleException(Parent as TComponent, ErrorMsg);
  end;

(*  procedure TAuThread.CallOnDone;
  begin
    if Assigned((Parent as TAuOutput).FOnDone) then
       (Parent as TAuOutput).FOnDone(Parent);
  end; *)     

{$IFDEF WIN32}
  procedure TAuOutput.Abort;
  begin
    TerminateThread(Thread.Handle, 0);
    WhenDone;
  end;
{$ENDIF}

  function TAuInput.CopyData;
  var
    P : Pointer;
  begin
    Result := BufferSize;
    GetData(P, Result);
    if P <> nil then
      Move(P^, Buffer^, Result);
  end;


end.


