(*
  This file is a part of New Audio Components package v 1.1
  Copyright (c) 2002-2007, Andrei Borovsky. All rights reserved.
  See the LICENSE file for more details.
  You can contact me at anb@symmetrica.net
*)

(* $Revision: 1.6 $ $Date: 2007/07/08 04:59:19 $ *)

unit ACS_smpeg;

interface

uses
  Classes, SysUtils, ACS_Types, ACS_Classes, mp3decdr;

const
  InitialBufferSize : Integer = $1000;

type

  TMP3In = class(TAuFileIn)
  private
    CurrentBufSize : Integer;
    FEOF : Boolean;
    mp3dec : mp3dec_t;
    FBitRate : Integer;
    FLayer : Integer;
    _Buffer : Pointer;
  protected
    procedure OpenFile; override;
    procedure CloseFile; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure GetData(var Buffer : Pointer; var Bytes : Integer); override;
    procedure Init; override;
    function Seek(SampleNum : Integer) : Boolean; override;
    property BitRate : Integer read FBitRate;
    property Layer : Integer read FLayer;
  end;

implementation

  function MP3Seek(cb_data : Pointer; offset : Integer; origin : Integer) : Integer; cdecl;
  var
    MP3In : TMP3In;
    SOR : Word;
  begin
    MP3In := TMP3In(cb_data);
    case origin of
      SEEK_CUR : SOR := soFromCurrent;
      SEEK_END : SOR := soFromEnd;
      SEEK_SET : SOR := soFromBeginning;
    end;
    Result := MP3In.Stream.Seek(offset, SOR);
  end;

  function MP3Read(cb_data : Pointer; buf : Pointer; bufsize : Integer) : Integer; cdecl;
  var
    MP3In : TMP3In;
    OldPos : Int64;
  begin
    MP3In := TMP3In(cb_data);
    OldPos := MP3In.Stream.Position;
    if OldPos >= MP3In.Stream.Size then
    begin
      Result := 0;
      Exit;
    end;
    Result := MP3In.Stream.Read(buf^, bufsize);
    if MP3In.Stream.Position > MP3In.Stream.Size then Result := MP3In.Stream.Size - OldPos;
  end;

  function MP3Close(cb_data : Pointer) : Integer; cdecl;
  var
    MP3In : TMP3In;
  begin
    MP3In := TMP3In(cb_data);
    if MP3In.Stream <> nil then MP3In.Stream.Free;
    MP3In.Stream := nil;
    Result := 0;
  end;


  constructor TMP3In.Create;
  begin
    inherited Create(AOwner);
    FEOF := True;
    CurrentBufSize := InitialBufferSize;
    if not (csDesigning	in ComponentState) then
    begin
      GetMem(_Buffer, CurrentBufSize);
      if not Libmp3decdrLoaded then
        raise EAuException.Create(Libmp3decdrPath + ' library could not be loaded.');
    end;
  end;

  destructor TMP3In.Destroy;
  begin
    if not (csDesigning	in ComponentState) then
      FreeMem(_Buffer);
    inherited Destroy;
  end;

  procedure TMP3In.Init;
  begin
    inherited Init;
  end;

  function TMP3In.Seek(SampleNum : Integer) : Boolean;
  begin
    if not Self.Busy then
      raise EAuException.Create('Cannot seek in closed stream.');
    Result := True;
    DataCS.Enter;
    Result := False;
    if mp3dec_seek(mp3dec, SampleNum, MP3DEC_SEEK_SAMPLES) = MP3DEC_RETCODE_OK then
    begin
      Result := True;
      FPosition := SampleNum*Self.FChan*(Self.FBPS div 8)
    end;
    DataCS.Leave;
  end;

  procedure TMP3In.OpenFile;
  var
    Size, res : Integer;
    SampleSize, Samples : Integer;
  begin
    if FOpened = 0 then
    begin
      FValid := True;
      if not FStreamAssigned then
        FStream := TFileStream.Create(FFileName, fmOpenRead);
      FStream.Seek(0, soFromEnd);
      Size := FStream.Position;
      FStream.Seek(0, soFromBeginning);
      mp3dec := mp3dec_init;
      mp3dec_set_config(mp3dec, MP3DEC_CONFIG_AUTO, MP3DEC_CONFIG_16BIT); // TODO : Add 24 bit support
      res := mp3dec_init_file(mp3dec, Self, MP3Seek, MP3Read, MP3Close, Size, 0);
      if res <> MP3DEC_RETCODE_OK then
        raise EAuException.Create('MP3 decoder setup failed.');
      res := mp3dec_get_stream_info(mp3dec, FChan,  FSR, SampleSize, Samples);
      if res <> MP3DEC_RETCODE_OK then
      begin
        Self.FValid := False;
        raise EAuException.Create('MP3 file doesn''t appear to be valid.');
      end;
      FBPS := (SampleSize div FChan) * 8;
      FTotalSamples := Samples;
      FTime := Samples div FSR;
      mp3dec_get_file_info(mp3dec, FBitRate, FLayer);
      FSeekable := True;
      FEOF := False;
    end;
    Inc(FOpened);
  end;

  procedure TMP3In.CloseFile;
  begin
    if FOpened = 1 then
    begin
      mp3dec_uninit(mp3dec);
      if not FStreamAssigned then
        if FStream <> nil then FStream.Free;
      FEOF := True;
    end;
    if FOpened > 0 then Dec(FOpened);
  end;

  procedure TMP3In.GetData(var Buffer : Pointer; var Bytes : Integer);
  var
    res, curtime, offs : Integer;
    used : LongWord;
  begin
    if FEOF then
    begin
      Buffer := nil;
      Bytes := 0;
      Exit;
    end;
    DataCS.Enter;
    if CurrentBufSize < Bytes then
    begin
      CurrentBufSize := Bytes;
      FreeMem(_Buffer);
      GetMem(_Buffer, CurrentBufSize);
    end;
    if FLoop then
      if FPosition >= FSize then
      begin
        FPosition := 0;
        mp3dec_seek(mp3dec, 0, MP3DEC_SEEK_SAMPLES);
      end;
    if Bytes > FSize - FPosition then
      Bytes := FSize - FPosition;
    res := mp3dec_decode(mp3dec, _Buffer, Bytes, used);
    if (used = 0) or (res <> MP3DEC_RETCODE_OK) then
    begin
      FEOF := True;
      Buffer := nil;
      Bytes := 0;
      DataCS.Leave;
      Exit;
    end;
    Bytes := used;
    Buffer := _Buffer;
    Inc(FPosition, Bytes);
    DataCS.Leave;
  end;

end.
